#from asyncio.tasks import async

LAUNCH_POWER = 140

from surrortg import Game
from surrortg.inputs import Switch, Joystick
import serial
import asyncio
import logging
from builtins import str
#ARDUINO = "/dev/serial/by-id/usb-Arduino__www.arduino.cc__0043_55739323637351B05162-if00" #servo demoon kytetty arduino
GUN = "/dev/serial/by-id/usb-Arduino__www.arduino.cc__0042_7573530393135180D0B2-if00" #gun arduino mega serial
TABLE = "/dev/serial/by-id/usb-Arduino__www.arduino.cc__0042_758343535303514191C0-if00" #table arduino mega serial

#table class that represents the cup positions for each team
class Table():
    
    def __init__(self, team):
        #team 1 or two
        self.team = team
        #Cup up/down position indicated by 1/0. Initialize cups to UP-position.
        self.cups = "000000"
        
    #change up/down cup position string by cup number
    def change_cup_pos(self, cup_num):
        #change to down position if already in up position, else to up position
        if self.cups[cup_num - 1] == '0':
            self.cups = self.cups[:cup_num - 1] + '1' + self.cups[cup_num:]

class AimSwitch(Switch):
    def __init__(self, ser):
        
        self.ser = ser
        self.lock = asyncio.Lock()
        
    #write t to drive aiming motor
    async def on(self):
        logging.info("aim on")
        self.ser.write(b't\n')

    #write anything to brake motor
    async def off(self):
        logging.info("aim off")
        self.ser.write(b'x\n')

class LaunchSwitch(Switch):
    def __init__(self, ser_gun, ser_table, team1, team2, current_team, io):
        self.team1 = team1
        self.team2 = team2
        self.current_team = current_team
        #init serial for gun arduino mega
        self.ser_gun = ser_gun
        #init serial for table arduino mega
        self.ser_table = ser_table
        self.io = io
        #self.score_function = score_function
        self.launched = False
        
        #launch command is v + launch power, combine into string and write to serial
    async def on(self):
        if not self.launched:
            self.launched = True
            #disable player input for the shooter
            if self.current_team.team == 1:
                self.io.disable_input(0)
            else:
                self.io.disable_input(1)
            
            #tell gun to stop moving
            self.ser_gun.write(b'x\n')
            
            await asyncio.sleep(1)
            
            logging.info("launch on: power ")
            launch_command = 'v' + str(LAUNCH_POWER)
            self.ser_gun.write(bytes(launch_command, encoding='ascii'))
            
            
            await self.table_event()
        else:
            logging.info("Already launched")
        #wait for 3 seconds before asking for hit or miss
        #await asyncio.sleep(3)
        #ask for result, if 0 (or something else), no hits reigstered, other number means hit and update cups
        #
        
    async def off(self):
        logging.info("launch off")
    
    async def table_event(self):
        #write detect command to arudino to start reading sensors
        logging.info("Writing detect")
        self.ser_table.write(b'd\n')
        logging.info("Waiting result")
        
        try:
            await asyncio.sleep(8)
            result_byte = self.ser_table.readline()
            result = int(result_byte.rstrip())
        except asyncio.CancelledError:
            logging.info("No answer, result is zero")
            result = 0
            
        
        
        #wait for the cup number that was hit or get 0 if no hit
        logging.info("Hit cup (zero if miss): ")
        
        #update table for current player
        if result >= 1 and result <= 6:
            self.current_team.change_cup_pos(result)
        logging.info("Team   cup positions: ")
        
        #check if all cups are down and it's time to send final score signaling game finish
        logging.info("Checking if current team wins")
        self.check_win()
        
        #change current team to the other team
        if self.current_team.team == 1:
            self.current_team = self.team2
        else:
            self.current_team = self.team1
        logging.info("Changing current team to team ")
        logging.info("Team cup positions: ")
            
        #pass arduino the up/down positions of current team
        logging.info("Passing team   positions to arduino")
        self.pass_positions()
        #read that arduino positioning is ready
        self.ser_table.readline()
        logging.info("Positions ready, enabling controls")
        
        self.launched = False
    
        #enable inputs for current player

        if self.current_team.team == 1:
            self.io.enable_input(0)
            self.io.set_current_seat(0)
        else:
            self.io.enable_input(1)
            self.io.set_current_seat(1)
          
    def pass_positions(self):
        self.ser_table.write(b't\n')
        self.ser_table.readline()
        logging.info("Ready-line read")
        position_string = self.current_team.cups + "\n"
        logging.info("Sending poisiton string: ")
        self.ser_table.write(bytes(position_string, encoding='ascii'))
        
    def check_win(self):
        if self.current_team.cups == "111111":
            #disable inputs
            self.io.disable_input(0)
            self.io.disable_input(1)
            #count scores for each player
            score_team1 = 0
            score_team2 = 0
            for c1 in self.team1.cups:
                if int(c1) == 1:
                    score_team1 += 100
                    
            for c2 in self.team2.cups:
                if int(c2) == 1:
                    score_team2 += 100
            
            #send final score for current (winner) seat
            if self.current_team.team == 1:
                self.io.send_score(score=score_team2, scores=None, seat=1, final_score=False)
                self.io.send_score(score=score_team1, scores=None, seat=0, final_score=True)
            else:
                self.io.send_score(score=score_team1, scores=None, seat=0, final_score=False)
                self.io.send_score(score=score_team2, scores=None, seat=1, final_score=True)

             
class PowerUpSwitch(Switch):
    def __init__(self, ser_table):
        self.ser_table = ser_table
        
    #increase launch power, 255 is maximum
    async def on(self):
        global LAUNCH_POWER
        
        #the number of previous leds that are lighted
        prev_leds = self.count_leds()
        logging.info("prev led: ")
        
        if LAUNCH_POWER + 3 > 255:
            LAUNCH_POWER = 255
        else:
            LAUNCH_POWER += 3
        #show launch power in terminal
        logging.info("power up on: ")
        
        now_leds = self.count_leds()
        logging.info("now_led")
        
        #if previous amount of leds is different, signal table arduino to light up leds
        if now_leds != prev_leds:
            self.signal_led(now_leds)
            logging.info("signaling leds")
    
    #function that counts the amount of leds indicating launch power
    def count_leds(self):
        #range of launch power
        p_range = 255 - 80
        #total count of leds
        led_total = 16
        #launch power per led lighted up
        power_per_led = p_range / led_total
        #normalizes launch power to start from 0 (-80)
        launch_scale = LAUNCH_POWER - 80
        #count of leds to light up, rounded up
        led_count = round(launch_scale / power_per_led)
        
        return led_count
        
        
    def signal_led(self, count):
            led_command = "l" + str(count) + "\n"
            self.ser_table.write(bytes(led_command, encoding='ascii'))
        
    async def off(self):
        logging.info("power up off")
        
class PowerDownSwitch(Switch):
    def __init__(self, ser_table):
        self.ser_table = ser_table
        
    #decrease launch power, 255 is maximum
    async def on(self):
        global LAUNCH_POWER
        
        #the number of previous leds that are lighted
        prev_leds = self.count_leds()
        
        if LAUNCH_POWER - 3 < 80:
            LAUNCH_POWER = 80
        else:
            LAUNCH_POWER -= 3
        #show launch power in terminal
        logging.info("power down on: ")
        
        now_leds = self.count_leds()
        
        #if previous amount of leds is different, signal table arduino to light up leds
        if now_leds != prev_leds:
            self.signal_led(now_leds)
        
    async def off(self):
        logging.info("power down off")
        
    #function that counts the amount of leds indicating launch power
    def count_leds(self):
        #range of launch power
        p_range = 255 - 80
        #total count of leds
        led_total = 16
        #launch power per led lighted up
        power_per_led = p_range / led_total
        #normalizes launch power to start from 0 (-80)
        launch_scale = LAUNCH_POWER - 80
        #count of leds to light up, rounded up
        led_count = round(launch_scale / power_per_led)
        
        return led_count
        
        
    def signal_led(self, count):
            led_command = "l" + str(count) + "\n"
            self.ser_table.write(bytes(led_command, encoding='ascii'))
   

class ABPGame(Game):
    async def on_init(self):
        #table object for both teams
        self.team1 = Table(1)
        self.team2 = Table(2)
        #current team variable for switching teams
        self.current_team = self.team1
        #init serial for gun arduino mega
        self.ser_gun = serial.Serial(GUN, baudrate = 115200)
        try:
            await asyncio.sleep(4)
            gun_line = self.ser_gun.readline()
            logging.info("On init:")
        except asyncio.CancelledError:
            logging.info("Didn't hear from serial")
        #init serial for table arduino mega
        self.ser_table = serial.Serial(TABLE, baudrate = 115200)
        try:
            await asyncio.sleep(4)
            table_line = self.ser_table.readline()
            logging.info("On init:")
        except asyncio.CancelledError:
            logging.info("Didn't hear from serial")
        
        #pass switches for object
        self.aim_switch = AimSwitch(self.ser_gun)
        self.launch_switch = LaunchSwitch(self.ser_gun, self.ser_table, self.team1, self.team2, self.current_team, self.io)
        self.power_up_switch = PowerUpSwitch(self.ser_table)
        self.power_down_switch = PowerDownSwitch(self.ser_table)
        
        #register inputs
        #seat 0 is team 1, seat 1 is team 2
        self.io.register_inputs({"aim": self.aim_switch, "launch": self.launch_switch, "power_up": self.power_up_switch, "power_down": self.power_down_switch}, seat=0)
        self.io.register_inputs({"aim": self.aim_switch, "launch": self.launch_switch, "power_up": self.power_up_switch, "power_down": self.power_down_switch}, seat=1) 
        
    async def on_prepare(self):
        #reset team cup positions
        self.team1.cups = "000000"
        self.team2.cups = "000000"
        #shoot ball command to move balls up the pipe and ensure balls are loaded
        self.ser_gun.write(b'a\n')
        #wait for table calibration
        self.calibration()
    
    #calibrate table function, cups calibrated to up position switch
    def calibration(self):
        self.ser_table.write(b'c\n')
        self.ser_table.readline()
        logging.info("Calibration successful")
    
    async def on_start(self):
        self.current_team = self.team1
        #enable team 1 controls
        self.io.enable_input(0)
        #disable team 2 controls
        self.io.disable_input(1)
        
        #current team is seat 0 / team 1
        self.io.set_current_seat(0)
        
        #play until final score is sent or timeout
        #game time inside sleep, timeout if game time reached 
        try:
            await asyncio.sleep(900)
            logging.info("ABSOLUTE_GAME_MAX_TIME passed, this should never happen")

            
            self.io.send_playing_ended()
        except asyncio.CancelledError:
            logging.info("GE ended playing")
            #count and send scores
            score_team1 = 0
            score_team2 = 0
            for c1 in self.team1.cups:
                if int(c1) == 1:
                    score_team1 += 100
                    
            for c2 in self.team2.cups:
                if int(c2) == 1:
                    score_team2 += 100
            
            self.io.send_score(score=score_team1, scores=None, seat=0, final_score=False)
            self.io.send_score(score=score_team2, scores=None, seat=1, final_score=False)
        
    #close serials on exit
    async def on_exit(self, *args, **kwargs):
        self.ser_gun.close()
        self.ser_table.close()

    
    
        
ABPGame().run()
