#File name: main.py
#Author: Antti Matikainen
#Date created: 6.7.2018
#Python version: 3.something

#This file contains the main function of the program and the ui.
#The UI is implemented using Tkinter as it is quite primitive.
#UI calls other functions in order to complete the functionality
#required of the datalogger.

#Imports and such
import sys
import os
import time
import errno
import serial.tools.list_ports
import multiprocessing
import queue
from tkinter import *

#Adding sstuff from subfolders
addPathString = os.path.join(os.getcwd(),'fileOperations')
sys.path.append(addPathString)
from writeStringToFile import writeStringToFile

addPathTherm = os.path.join(os.getcwd(), 'rasPiColdsideCompensation/Thermocouple')
sys.path.append(addPathTherm)
addPathOp = os.path.join(os.getcwd(), 'rasPiColdsideCompensation')
sys.path.append(addPathOp)
from coldsideCompensation import coldsideCompensation

#Function for the sensor reading process.
#Should probably be in a separate file but desperate times.
def sensorReadingFunction(ports, q, targetPath):
  #Serial port opening and so on here.
  portsToRead = []
  for x in ports:
    portsToRead.append(serial.Serial(x.device))

  #Set a variable that can be toggled easily to shut down the loop
  variable = True
  while variable:
    #Trying to read from the queue used to send the shutdown signal.
    #Catching the queue.Empty exception allows us to simply continue
    #as normal in situations where the loop has not been shut down.
    try: readvar = q.get_nowait()
    except queue.Empty:
      #The order to shut the process down has not been given.
      #Perform sensor functions. Format a string to write to file.
      #Flushing ports in order to get up to date readings.
      for porttoflush in portsToRead:
        porttoflush.flush()
      #Creating empty list to add readings to.
      tmpvars = []
      #Reading data from ports.
      for porttoread in portsToRead:
        tmpvars.append(porttoread.readline())
        #print(tmpvars)
      #Handling readings, converting and so on.
      for datapoint in tmpvars:
        #Convert string containing readnig into floating point value
        datavar = float(datapoint)
        print(str(datavar))
        #Convert floating point voltage into temperature
        datavar = coldsideCompensation(datavar, 0, 'K')
        print(str(datavar))
      #Formatting string
      logString = str(time.time()) + " test var: " + str(tmpvars) + "\n"
      #Writing to file.
      writeStringToFile(targetPath, logString)
      #Sleep for 1 second as a crude timing mechanism. Bad for now, will be fix
      time.sleep(1)
    else:
      if readvar == "Shutdown":
        variable = False
        for x in portsToRead:
          x.close()

class App:
  def __init__(self, master):
    #Creating a Frame widget to contain button widgets.
    self.buttonFrame = Frame(master)
    self.buttonFrame.grid(row=0)

    #Creating a second frame widget to contain stuff below the buttons.
    self.textBoxFrame = Frame(master)
    self.textBoxFrame.grid(row=1)

    #Creating a frame for the comment entry box and button.
    self.commentFrame = Frame(master)
    self.commentFrame.grid(row=2)

    #Creating buttons to control the program.
    self.initButton = Button(self.buttonFrame, text="1. Re-initialize logger", fg="black", command=self.initFunction)
    self.scanButton = Button(self.buttonFrame, text="2. Scan for sensors", fg="black", command=self.scanFunction)
    self.startRecordingButton = Button(self.buttonFrame, text="3. Begin recording", fg="green", command=self.startRecordingFunction)
    self.stopRecordingButton = Button(self.buttonFrame, text="4. Stop recording", fg="red", command=self.stopRecordingFunction)
#    self.visualizeDataButton = Button(self.buttonFrame, text="6. Visualize data", fg="black", command=self.visualizeDataFunction)

    #Packing buttons as required by Tk.
    self.initButton.grid(row=0, column=1)
    self.scanButton.grid(row=0, column=2)
    self.startRecordingButton.grid(row=0, column=4)
    self.stopRecordingButton.grid(row=0, column=5)
#    self.visualizeDataButton.grid(row=0, column=6)

    #Creating a little display to inform the user about the current
    #state of the program so they know which button to press next.
    self.v = StringVar()
    self.stateLabel = Label(self.buttonFrame, textvariable=self.v)
    self.stateLabel.grid(row=0, column=7)
    self.v.set("Current state: 0")

    #Creating a text box widget to contain function diagnostics printouts.
    self.infoTextBox = Text(self.textBoxFrame, height=10, width=80)
    self.infoTextBox.grid(row=1, column=1)

    #Create a scrollbar to make text box easier to use.
    self.infoBoxScrollbar = Scrollbar(self.textBoxFrame)
    self.infoBoxScrollbar.grid(row=1, column=2, sticky=N+S)

    #Using Tkinter foncig commandsto make scrollbar work with text box.
    self.infoBoxScrollbar.config(command = self.infoTextBox.yview)
    self.infoTextBox.config(yscrollcommand=self.infoBoxScrollbar.set)

    #Creating an entry widget to allow user to input comments.
    self.userCommentEntry = Entry(self.commentFrame)
    self.userCommentEntry.grid(row=2, column=1)

    self.userCommentEntry.focus_set()

    #Create button that is used to enter the text.
    self.commentEntryButton = Button(self.commentFrame, text="Enter comment", command=self.textEntryFunction)
    self.commentEntryButton.grid(row=2, column=2)

    #Adding red/green box that is based on recording state.
    #Allows user to tell state easier.
    #Creating label.
    self.stateColorLabel = Label(self.commentFrame, text="Red/Green based on recording state:")
    self.stateColorLabel.grid(row=2, column=4)
    #Creating canvas.
    self.stateColorCanvas = Canvas(self.commentFrame, width=25, height=25)
    self.stateColorCanvas.grid(row=2, column=5)
    #Creating colored rectangle.
    self.stateColorSquare = self.stateColorCanvas.create_rectangle(0, 0, 50, 50, fill="red")

    #A variable used to control the state of the program.
    #Ensures buttons are pressed in order. Disabling buttons
    #or making them invisible not doable to my knowledge.
    #0 initial state
    #1 Initialized, name entered, folders/files created.
    #2 Serial ports scanned
    #3 Recording data.
    self.stateInt = 0

    #Calling the function to initialize the name.
    #Same function can also be called on from the button if need be.
    self.initFunction()

  #Functions called by the buttons created during the App constructor.
  #Initial version contains test prints, will be expanded.
  def initFunction(self):
    if (self.stateInt == 0 or self.stateInt == 4):
      #Inserting diagnostics info into text box.
      self.infoTextBox.insert(END, "Beginning datalogger initialization.\n")
      #self.infoTextBox.insert(END, "Current time: " + self.time + ".\n")

      #Creating a pop-up window to take user input.
      self.userInputPopUp = Toplevel()
      self.userInputPopUp.title("Name input")

      #Adding informative text to popup.
      self.msg = Label(self.userInputPopUp, text="Please enter a name to be combnied with the date.")
      self.msg.grid(row=0, column=1)

      #Adding text box to take user input.
      self.popUpTextBox = Entry(self.userInputPopUp)
      self.popUpTextBox.grid(row=1, column=1)

      #Adding button for saving input and removing the box.
      self.popUpButton = Button(self.userInputPopUp, text="Save Name", command=self.popUpFunction)
      self.popUpButton.grid(row=2, column=1)
    else:
      self.infoTextBox.insert(END, "Incorrect state for operation. State: " + str(self.stateInt) + "\n")

  #Function for evaluating user input for the name.
  def popUpFunction(self):
      #Reading entry from text box and evaluating it.
      #Currently only checks for empty entry. Can be expanded.
      tempTextVar=self.popUpTextBox.get()
      if (tempTextVar.strip()):
        #Outputting diagnostics, setting userNameString variable, destroying popup.
        #userNameString is combined with time to form the folder and file names.
        self.infoTextBox.insert(END, "User inputted name for experiment is: " + tempTextVar + "\n")
        self.userNameString = tempTextVar
        self.userInputPopUp.destroy()

        #Taking the time. Combined with the name given by the user.
        self.time=time.asctime(time.localtime())

        #Combining time and user input into a name for the file/folder.
        self.fullTimeNameString = (self.time + "-" + self.userNameString).replace(" ","")

        #Creating the folder to be used for the data and comment files.
        #Geting the current location and forming a path.
        currentLocation = os.path.dirname(os.path.abspath(__file__))
        self.filePath = currentLocation + os.path.dirname("/tmp/" + self.fullTimeNameString)

        #Using try except to check if the directory exists before making it.
        try:
          os.makedirs(self.filePath)
        except OSError as exception:
          if exception.errno != errno.EEXIST:
            raise

        #Creating header info on the data file.
        writeStringToFile(self.filePath + "/" + self.fullTimeNameString, "# Thermocouple Logger File \n")

        #Setting variable, updating label.
        self.stateInt = 1
        self.v.set("Current state: " + str(self.stateInt) + "\n")

      else:
        #Outputting diagnostics, allowing for user to enter a new name.
        self.infoTextBox.insert(END, "Invalid input for experiment name.\n")

  def scanFunction(self):
    if (self.stateInt == 1):
      #Setting state variable and updating label.
      self.stateInt = 2
      self.v.set("Current state: " + str(self.stateInt) + "\n")
      #Outputting diagnostics.
      self.infoTextBox.insert(END, "Beginning sensor scan function.\n")

      #Calling function.
      #Function from the PySerial library that returns a list of ports.
      self.listPorts = []
      tmpPorts = serial.tools.list_ports.comports()

      #Giving a list of all the serial ports so that they can
      #be added to the .txt by the user.
      self.infoTextBox.insert(END, "Found the following:\n")
      for meh in tmpPorts:
        self.infoTextBox.insert(END, meh.name + "\n")

      #Parsing a text file containing the names of devices to read.
      #Parsin it to complete a list of sensors.
      portFile = open(os.path.join(sys.path[0], 'sensorstoread.txt'), 'r')
      desiredports = portFile.read().splitlines()
      #print(desiredports)
      portFile.close()
      #Adding desired ports to list of ports to read based on list from file.
      for port in tmpPorts:
        if port.name in desiredports:
          self.listPorts.append(port)
          #print(port.name)
      #Writing info to text box and file.
      self.infoTextBox.insert(END, "Utilizing the following:\n")
      for thingy in self.listPorts:
        self.infoTextBox.insert(END, thingy.name + "\n")

      writeStringToFile(self.filePath+"/"+self.fullTimeNameString, "# Serial port sensor count: " + str(len(self.listPorts)) + "\n")
      
    else:
      self.infoTextBox.insert(END, "Incorrect state for operation. State: " + str(self.stateInt) + "\n")

  def startRecordingFunction(self):
    if (self.stateInt == 2):
      #Setting state integer and updatnig label.
      self.stateInt = 3
      self.v.set("Current state: " + str(self.stateInt) + "\n")
      #Outputting diagnostics.
      self.infoTextBox.insert(END, "Beginning sensor data recording function.\n")
      #Changing square color
      self.stateColorCanvas.itemconfig(self.stateColorSquare, fill="green")
      #Calling function.
      #Creating a queue for interprocess communication.
      self.commQueue = multiprocessing.Queue(1)
      #Creating the multiprocessing.Process object and keeping it in variable.
      self.p = multiprocessing.Process(target=sensorReadingFunction, args=(self.listPorts,self.commQueue,self.filePath+"/"+self.fullTimeNameString,))
      #Starting the process.
      self.p.start() 

    else:
      self.infoTextBox.insert(END, "Incorrect state for operation. State: " + str(self.stateInt) + "\n")

  def stopRecordingFunction(self):
    if (self.stateInt == 3):
      #Setting state and updating label.
      self.stateInt = 4
      self.v.set("Current state: " + str(self.stateInt) + "\n")
      #Outputting diagnostics.
      self.infoTextBox.insert(END, "Beginning stop data recording function.\n")
      #Setting color square to red.
      self.stateColorCanvas.itemconfig(self.stateColorSquare, fill="red")
      #Calling function.
      #Writing exit command to queue
      self.commQueue.put("Shutdown")
      self.p.join()
      #Inform the user that recording stopped successfully.
      self.infoTextBox.insert(END, "Recording closed successfully.\n")
    else:
      self.infoTextBox.insert(END, "Incorrect state for operation. State: " + str(self.stateInt) + "\n")

  def textEntryFunction(self):
    if not self.stateInt == 0:
      tmpcomment = self.userCommentEntry.get()+"\n"
      #Outputting diagnostics.
      self.infoTextBox.insert(END, "Writing comment to file: " + tmpcomment)
      #Calling function.
      if not writeStringToFile( self.filePath + "/" + self.fullTimeNameString + ".log",tmpcomment):
        self.infoTextBox.insert(END, "Failed to write comment.")

# COMMENTED OUT AS WE RAN OUT OF TIME
#  def visualizeDataFunction(self):
#    if (self.stateInt == 4):
#      #Outputting diagnostics.
#      self.infoTextBox.insert(END, "Visualizing data.\n")
#
#      #Calling function.
#
#    else:
#      self.infoTextBox.insert(END, "Incorrect state for operation. State: " + str(self.stateInt) + "\n")

def main():
  #If check related to multiprocessing.
  if __name__ == '__main__':
    #Creating the root widget required by Tk functionality.
    root = Tk()
    app = App(root)

    #Setting the multiprocessing start method to spawn.
    #Allows for use on UNIX and Windows without headaches.
    multiprocessing.set_start_method('spawn')

    #Going into the Tk main loop which handles all functionality.
    root.mainloop()

#Calling function.
main()
